﻿################################################################################
# 
# checksum ver.1.00.00 (20240720a)
# 
# チェックサムを表示または照合する
# Print or check checksums
# 
# 対応アルゴリズム Algorithms
# MD5 / SHA1 / SHA256 / SHA384/ SHA512 / cksum CRC-32 / sum (BSD) / sum (SystemV)
# 
# ------------------------------------------------------------------------------
#  作者  Auther
# --------------
# ritsuka
# 
#   Web                    https://ritska.com
#   twitter                @ritsukaPya
#   ActivityPub (misskey)  @ritsukaPya@misskey.io
#   Bluesky                @ritsukapya.bsky.social
# 
# ------------------------------------------------------------------------------
#  ライセンス  License
# ---------------------
# このソフトウェアは「二条項BSDライセンス」の下に公開されています。
# 詳細は下記のファイルを参照して下さい。
#  * README_ja.txt
#  * README_en.txt
# 
# This software is provided under the "2-clause BSD license".
# See also the files.
#  * README_ja.txt
#  * README_en.txt
# 
# Copyright (C) 2024 ritsuka
# 
################################################################################

using namespace System.Collections.Generic;

param(
	# [ValidateSet("md5", "sha1", "sha256", "sha384", "sha512", "rmd160", "cksum", "sumbsd", "sumsysv")]
	[ValidateSet("md5", "sha1", "sha256", "sha384", "sha512", "cksum", "sumbsd", "sumsysv")]
	[Alias("a")][string] $algorithm = "sha256", # ハッシュアルゴリズム

	[Alias("bsd")][switch] $tag,       # BSD形式で出力

	[Alias("o")][string] $outfile="",  # 出力先ファイル名

	[Alias("f")][switch] $force,       # 出力先ファイルが存在した場合、強制的に上書きする

	            [switch] $add,         # 出力先ファイルが存在した場合、末尾に追記する

	[Parameter(Position=0)]
	            [string] $target="",   # 対象ファイル名（ワイルドカード指定可）

	[Alias("r")][switch] $recurse,     # カレントディレクトリ以下を再帰検索

	[Alias("z")][switch] $zero,        # 区切りに改行ではなくNULを使い、ファイル名をエスケープしない（エスケープはWindowsではもともとない）

	[Alias("c")][string] $check="",    # 検証モード（チェックサムファイル名を指定）

	# バイナリモードを指定した時に結果出力に*印がつく以外何もしない デフォルトでテキストモード 両指定時はバイナリモード
	[Alias("b")][switch] $binary,
	[Alias("t")][switch] $text,

	[Alias("e")][switch] $english,     # ユーザーのロケールに関わらずメッセージを英語表示する

	[Alias("usage")][switch] $help,
	[Alias("v")][switch] $version,

	# パイプライン 標準入力(UTF-8文字列 または byte[])
	[Parameter(ValueFromPipeline=$True)]$stdin
);


# --------------------------------------------------------------------
# VERSION AND AUTHER
$COM_NAME   = "checksum";
$COM_VER1   = "1.00.00";
$COM_VER2   = "20240720a";
$COM_AUTHER = "ritsuka"


# --------------------------------------------------------------------
# HELP MESSAGE
$HELP_EN = @"
Usage: ${COM_NAME} [OPTION]... [FILE]...
Print or check MD5/SHA1,256,384,512/CKSUM/SUM(BSD,SystemV) checksums.

Example:
  Display SHA256 checksums of files
    ${COM_NAME} *.jpg
  Display MD5 checksums of files
    ${COM_NAME} -a md5 *.jpg
  Display SHA256 checksums of files (recursive search in current dir)
    ${COM_NAME} -r *.jpg
  Display SHA256 checksums of files and save result to file (LF,UTF-8 text)
    ${COM_NAME} *.jpg -o CHECKSUM.txt
  Verify checksums
    ${COM_NAME} -c CHECKSUM.txt

With no FILE, or when FILE is -, read pipeline as text (UTF-8) or as byte[].
  Example (pipeline):
    Get-Content textfile.txt | ${COM_NAME} <read as UTF-8 text>
    Get-Content -AsByteStream -Raw textfile.txt | ${COM_NAME} <read as byte[]>
  CAUTION:
    PowerShell's pipeline receives binary data as UTF-8 string.
    And the data will be DESTROYED. It is a specification.

Create checksums:
  -a, -algorithm TYPE  select the digest type to use.  See TYPE below.
  -r, -recurse         search current dir recursive (cannot set path include dir)
  -b, -binary          read in binary mode (equal text mode, add `"*`" flag only)
  -t, -text            read in text mode (default)
  -z, -zero            end each output line with NUL, not newline
      -tag, -bsd       create a BSD-style checksum

  -o, -outfile OUTFILE output to text file (LF, UTF-8 without BOM)
  -f, -force           overwrite text file, if exist (use with -o)
      -add             add to text file, if exist    (use with -o)

Verify checksums:
  -c, -check FILE      read checksums from the FILEs and check them

Other:
      -help, -usage    display this help and exit
  -v, -version         output version information and exit
  -e, -english         print message in English regardless of user's locale

TYPE determines the digest algorithm:
  md5     : MD5 (128-bit)
  sha1    : SHA1 (160-bit)
  sha256  : SHA256 (256-bit) *DEFAULT*
  sha384  : SHA384 (384-bit)
  sha512  : SHA512 (512-bit)
  sumsysv : equivalent to UN*X "sum" command - System V algorithm
  sumbsd  : equivalent to UN*X "sum" command - BSD algorithm
  cksum   : equivalent to UN*X "cksum" command - CRC-32 algorithm

  NOTE: UN*X cksum command's algorithm is not plain CRC-32.
        Do not use this algorithm for calculating plain CRC-32 values.
"@;

$HELP_JA_JP = @"
使用法: ${COM_NAME} [オプション]... [ファイル]...
MD5/SHA1,256,384,512/CKSUM/SUM(BSD,SystemV) チェックサムを表示または照合します。

使用例:
  対象ファイルのSHA256ハッシュを表示
    ${COM_NAME} *.jpg
  対象ファイルのMD5ハッシュを表示
    ${COM_NAME} -a md5 *.jpg
  対象ファイルのSHA256ハッシュを表示 (カレントディレクトリ内を再帰検索)
    ${COM_NAME} -r *.jpg
  対象ファイルのSHA256ハッシュを表示 (結果をファイルに書き出し: LF,UTF-8 text)
    ${COM_NAME} *.jpg -o CHECKSUM.txt
  チェックサムファイルに記されたハッシュ値とファイルのハッシュ値を照合
    ${COM_NAME} -c CHECKSUM.txt

ファイルの指定がない場合やファイルが - の場合、パイプラインから読み込みを行います。
  (UTF-8文字列またはバイト配列 byte[] として解釈します)
  パイプライン読込の例:
    Get-Content textfile.txt | ${COM_NAME} <UTF-8文字列として読込>
    Get-Content -AsByteStream -Raw textfile.txt | ${COM_NAME} <byte[]として読込>
  注意:
    PowerShellのパイプラインは、ネイティブコマンド等からバイナリデータを受け取った際に
    UTF-8文字列への変換を試み、その結果データが壊れます。これは仕様です。

チェックサム作成:
  -a, -algorithm TYPE  使用するダイジェスト種別を指定する (下記の TYPE の説明を参照)
  -r, -recurse         現在のディレクトリ内を再帰検索する (ディレクトリを含むパスは指定不可)
  -b, -binary          バイナリモードで読み込む
                         (テキストモードとの違いはないが、出力に"*"マークがつく)
  -t, -text            テキストモードで読み込む (デフォルト)
  -z, -zero            出力行の区切りとして改行文字ではなくNULを使用する
      -tag, -bsd       BSD形式のチェックサムを作成する

  -o, -outfile OUTFILE テキストファイルに保存する (改行LF, BOMなしUTF-8)
  -f, -force           出力先ファイルが存在する場合に上書きする (-oと共に使用)
      -add             出力先ファイルが存在する場合に追記する   (-oと共に使用)

チェックサム照合:
  -c, -check FILE      FILEからチェックサムを読み込み、照合する

その他:
      -help, -usage    このヘルプを表示して終了する
  -v, -version         バージョン情報を表示して終了する
  -e, -english         ユーザーのロケールに関わりなくメッセージを英語表示にする

TYPE 使用可能なダイジェストアルゴリズム:
  md5     : MD5 (128-bit)
  sha1    : SHA1 (160-bit)
  sha256  : SHA256 (256-bit) *デフォルト*
  sha384  : SHA384 (384-bit)
  sha512  : SHA512 (512-bit)
  sumsysv : UN*Xの "sum" コマンドに相当 - System V方式アルゴリズム
  sumbsd  : UN*Xの "sum" コマンドに相当 - BSD方式アルゴリズム
  cksum   : UN*Xの "cksum" コマンドに相当 - cksum式CRC-32アルゴリズム

  NOTE: UN*Xのcksumで使用されているアルゴリズムは純粋なCRC-32とは異なります。
        純粋なCRC-32値を求めたい時には使用すべきではありません。
"@;

# --------------------------------------------------------------------
$INVALID_CHARS = [IO.Path]::GetInvalidFileNameChars() -ne '\' -ne '/'; # 具体的なパス名として使えない文字
$NONCUL = [cultureinfo]::InvariantCulture; # ToLowerなどで使用 どのカルチャ(言語)でもないことを示す
$USER_LOCALE = (Get-Culture).Name;
if($english){
	$USER_LOCALE = "en-US";
}


# --------------------------------------------------------------------
$isHelpMode    = [bool]$help;
$isVersionMode = [bool]$version;

$isFileOutput = $outfile -ne "";
$strOutFile = $outfile;
$isWriteForce = [bool]$force;
$isWriteAdd   = [bool]$add;

$isBsdStyleOutput = [bool] $tag;
$isRecurse        = [bool] $recurse;
$isBinMode        = [bool] $binary;
$isNullCharSep    = [bool] $zero;

$strBinFlag = " ";
if($isBinMode){
	$strBinFlag = "*";
}

$strAlgo = $algorithm.ToLower($NONCUL); # ハッシュアルゴリズム（小文字）

$strSeparater = "";
if($isNullCharSep){
	$strSeparater = [char]0x00;
}else{
	$strSeparater = "`n";
}

$isVerifyMode = $check -ne "";
$strSumFiles = $check;

$strSentence = $target; # 入力されたファイル指定用の文字列
if($isRecurse -and ($strSentence -eq "")){
	$strSentence = "*";
}

$isPipelineMode = ($null -ne $stdin) -and ($strSentence -match "^-?$"); # パイプライン入力あり かつ ファイル名指定なし の場合(両方ある場合ファイル優先)


# --------------------------------------------------------------------
# ハッシュアルゴリズム名
#   unixname: ハッシュファイル中に記載されるアルゴリズム名
#   msparam:  Get-FileHashに渡すアルゴリズム名
#   length:   ハッシュの文字列長(cksumなどは0埋めのない10進数であり文字列長が一定でないので0とする)
$ALGO_NAMES = @{
	md5      = @{ unixname="MD5";    msparam="MD5";    length=32  };
	sha1     = @{ unixname="SHA1";   msparam="SHA1";   length=40  };
	sha256   = @{ unixname="SHA256"; msparam="SHA256"; length=64  };
	sha384   = @{ unixname="SHA384"; msparam="SHA384"; length=96  };
	sha512   = @{ unixname="SHA512"; msparam="SHA512"; length=128 };
	#rmd160   = @{ unixname="RMD160"; msparam="RIPEMD160"; length=40  };
	cksum    = @{ unixname="";       msparam="";       length=0  };
	sumbsd   = @{ unixname="";       msparam="";       length=0  };
	sumsysv  = @{ unixname="";       msparam="";       length=0  };
};


# --------------------------------------------------------------------
# C#のクラスとそれを呼び出すためのPowershellラッパー関数
#   注意：Add-TypeしたC#のクラスをPowerShellのクラス内から直接呼ぶとエラーになる
#   （Add-Type行の位置に関わらずPowerShellクラスの解釈が先に走るため、存在しないクラスとみなされる）
#   PowerShellクラス内からはラッパー関数を呼ぶこと

[string]$strCsCode = @'
// ---------- C#コード ここから ----------
using System;
using System.IO;
using System.Collections.Generic;

public static class LoCsClass{

	//////////////////////////////////////////////////////////////////
	// sumコマンド(System Vアルゴリズム)準拠のチェックサムを求める
	//   戻値: (string)0～65535 ※0埋めなし
	//   ※ 注意 チェックサムファイル内の数値は0埋めされている可能性を考える 比較時は注意
	public static string CalcSumSysv(Stream stream){
		Int32 intArrSizeMax = 64*1024*1024; // 0～2,147,483,591
		UInt64 intTotal = 0x00;
		
		while(1 < stream.Length - stream.Position){
			Int32 intArrSize = (Int32) Math.Min(intArrSizeMax, (stream.Length - stream.Position));
			byte[] arrBytesTmp = new byte[intArrSize];
			stream.Read(arrBytesTmp, 0, intArrSize);

			foreach(byte b in arrBytesTmp){
				intTotal += b;
			}
		}

		UInt64 intR = (intTotal & 0xFFFF) + ((intTotal & 0xFFFFFFFF) >> 16);
		UInt32 intSumsysv = (UInt32) ((intR & 0xFFFF) + (intR >> 16));

		return Convert.ToString(intSumsysv);
	}


	//////////////////////////////////////////////////////////////////
	// sumコマンド(BSDアルゴリズム)準拠のチェックサムを求める
	//   戻値: (string)0～65535 ※0埋めなし
	//   ※ 参考 UbuntuのsumコマンドのBSDアルゴリズムに限り0埋めありの結果を返してくる
	//   ※ 注意 チェックサムファイル内の数値は0埋めされている場合あり 比較時は注意
	public static string CalcSumBsd(Stream stream){
		Int32  intArrSizeMax = 64*1024*1024; // 0～2,147,483,591
		UInt32 intSumBsd = 0x00;

		while(1 < stream.Length - stream.Position){
			Int32 intArrSize = (Int32) Math.Min(intArrSizeMax, (stream.Length - stream.Position));
			byte[] arrBytesTmp = new byte[intArrSize];
			stream.Read(arrBytesTmp, 0, intArrSize);
			
			foreach(byte b in arrBytesTmp){
				intSumBsd = (intSumBsd >> 1) + ((intSumBsd & 0x01) << 15);
				intSumBsd += b;
				intSumBsd = intSumBsd & 0xFFFF;
			}
		}

		return Convert.ToString((UInt16)intSumBsd);
	}


	//////////////////////////////////////////////////////////////////
	// cksumコマンド準拠のCRC-32値を求める
	//   単純なCRC-32計算ではなく、データサイズ(byte)数値をデータ末尾に付加してから
	//   計算している点に注意。
	//   戻値: (string)0～4294967295 ※0埋めなし
	//   ※ 注意 チェックサムファイル内の数値は0埋めされている可能性を考える 比較時は注意
	public static string CalcCkSum(Stream stream){
		Int32 intArrSizeMax = 64*1024*1024; // 0～2,147,483,591

		// 生成多項式: G(x) = x^32 + x^26 + x^23 + x^22 + x^16 + x^12 + x^11 + x^10 + x^8 + x^7 + x^5 + x^4 + x^2 + x + 1
		UInt32 intPolynomial = 0x04C11DB7; // 生成多項式（最上位1bitは0化し32bitに切り詰めたもの）
		UInt32 intCrc        = 0x00000000; // CRC初期値

		UInt64 intDataSize = (UInt64) stream.Length;

		// データ部のbyte数を表す。データ末尾に付加される（小さい桁が冒頭にくる逆順形式。配列後方の0はあとで省く）
		List<byte> glPostBytes = new List<byte>();
		glPostBytes.Add( (byte)( (intDataSize         ) & 0xFF ) );
		glPostBytes.Add( (byte)( (intDataSize >>  8   ) & 0xFF ) );
		glPostBytes.Add( (byte)( (intDataSize >> (8*2)) & 0xFF ) );
		glPostBytes.Add( (byte)( (intDataSize >> (8*3)) & 0xFF ) );
		glPostBytes.Add( (byte)( (intDataSize >> (8*4)) & 0xFF ) );
		glPostBytes.Add( (byte)( (intDataSize >> (8*5)) & 0xFF ) );
		glPostBytes.Add( (byte)( (intDataSize >> (8*6)) & 0xFF ) );
		glPostBytes.Add( (byte)( (intDataSize >> (8*7)) & 0xFF ) );
		// 末尾のゼロを省く
		for(int i=(glPostBytes.Count - 1); 0<i; i--){
			if(glPostBytes[i] == 0x00){
				glPostBytes.RemoveAt(i);
			}else{
				break;
			}
		}
		// 末尾に0を4byte(32bit: 生成多項式の次数である32)を追加
		glPostBytes.AddRange( new List<byte>(){0x00, 0x00, 0x00, 0x00} );

		while(1 < stream.Length - stream.Position){
			Int32 intArrSize = (Int32) Math.Min(intArrSizeMax, (stream.Length - stream.Position));
			byte[] arrBytesTmp = new byte[intArrSize];
			stream.Read(arrBytesTmp, 0, intArrSize);
			
			foreach(byte b in arrBytesTmp){
				for(int i=0; i<8; i++){
					bool isOverflowOne = Convert.ToBoolean(intCrc & 0x80000000); // 最上位ビットが1かどうか(シフト時にあふれるのが1か)
					intCrc = intCrc << 1;
					intCrc = (UInt32) (intCrc | (UInt32)((b >> (7-i)) & 0x01)); // glBytesから最上位ビットを入れる
					if(isOverflowOne){
						intCrc = intCrc ^ intPolynomial;
					}
				}
			}
		}

		foreach(byte b in glPostBytes){
			for(int i=0; i<8; i++){
				bool isOverflowOne = Convert.ToBoolean(intCrc & 0x80000000); // 最上位ビットが1かどうか(シフト時にあふれるのが1か)
				intCrc = intCrc << 1;
				intCrc = (UInt32) (intCrc | (UInt32)((b >> (7-i)) & 0x01)); // glBytesから最上位ビットを入れる
				if(isOverflowOne){
					intCrc = intCrc ^ intPolynomial;
				}
			}
		}

		return Convert.ToString(~ intCrc);
	}


}
// ---------- C#コード ここまで ----------
'@;

function cs-CalcSumSysv([System.IO.Stream]$stream){
	Add-Type -TypeDefinition $strCsCode -Language CSharp;
	return [LoCsClass]::CalcSumSysv($stream);
}

function cs-CalcSumBsd([System.IO.Stream]$stream){
	Add-Type -TypeDefinition $strCsCode -Language CSharp;
	return [LoCsClass]::CalcSumBsd($stream);
}

function cs-CalcCkSum([System.IO.Stream]$stream){
	Add-Type -TypeDefinition $strCsCode -Language CSharp;
	return [LoCsClass]::CalcCkSum($stream);
}


# --------------------------------------------------------------------
class LoSum{

	##################################################################
	# テキストファイル書出（UTF-8 BOMなし）
	#   引数1: [string]書き出す文字列
	#   引数2: [string]出力ファイルパス
	#   引数3: ファイルが既存の場合上書きするか true:上書き  false:ファイル末尾に追記
	#   戻値 : なし
	static [void] WriteTextFile([string]$strContent, [string]$strOutFilePath, [bool]$isOverWrite=$false){
		$bytes = [Text.Encoding]::UTF8.GetBytes($strContent);

		$filemode = [System.IO.FileMode]::Append; # 末尾追記モード（ない場合新規作成）
		if($isOverWrite){
			$filemode = [System.IO.FileMode]::Create; # 上書きモード（ない場合新規作成）
		}

		$filestream = [System.IO.File]::Open($strOutFilePath, $filemode);
		foreach($byte in $bytes){
			$filestream.WriteByte($byte);
		}
		$filestream.close();
	}


	##################################################################
	# ファイル名・相対パスなどの文字列からフルパスを得る
	#   引数1: [string]ファイル名など
	#   戻値 : [string]フルパス（空文字を入れた場合・親ディレクトリが存在しない場合は空文字を戻す）
	static [string] GetFullpath([string]$strPath){
		if($strPath -eq ""){
			return "";
		}
		$filename = Split-Path $strPath -Leaf;   # 現在存在していなくてもよい
		$dirname  = Split-Path $strPath -Parent; # 存在している必要がある(Convert-Pathの都合)
		if($dirname -eq ""){
			$dirname = ".";
		}

		if(!(Test-Path -LiteralPath $dirname)){
			return "";
		}
		return (Join-Path (Convert-Path $dirname) $filename);
	}


	##################################################################
	# ロケールに応じたメッセージを選択する
	#   引数1: [Hashtable] @{ja_JP="メッセージ"; en="Message"; default="en"}
	#   戻値 : [string]
	static [string] SelectIntlMsg([Hashtable]$htMsg){
		$strLocaleLong  = $Script:USER_LOCALE;
		$strLocaleShort = $strLocaleLong -replace "-.*$", "";

		# 1. 例:en-USを探す
		$strMsg = $htMsg[$strLocaleLong];

		# 2. 例:enを探す
		if($null -eq $strMsg){
			$strMsg = $htMsg[$strLocaleShort];
		}

		# 3. 例:en-**を探す
		if($null -eq $strMsg){
			$arrMatchedKeys = @()
			foreach($key in $htMsg.Keys){
				if($strLocaleShort -match ($key -replace "_.*$", "")){
					$arrMatchedKeys += $key;
				}
			}
			if($arrMatchedKeys.length -ne 0){
				$strMsg = $htMsg[ ($arrMatchedKeys | Sort-Object)[0] ];
			}
		}

		# 4. defaultを使う
		if($null -eq $strMsg){
			$strMsg = $htMsg[$htMsg["default"]];
		}

		return [string]$strMsg;
	}


	##################################################################
	# cksumコマンド準拠のCRC-32値を求める
	#   単純なCRC-32計算ではなく、データサイズ(byte)数値をデータ末尾に付加してから
	#   計算している点に注意。
	#   戻値: [string]0～4294967295 ※0埋めなし
	#   ※ 注意 チェックサムファイル内の数値は0埋めされている可能性を考える 比較時は注意
	static [string] CalcCkSum([System.IO.Stream]$stream){
		return cs-CalcCkSum $stream;
	}


	##################################################################
	# sumコマンド(BSDアルゴリズム)準拠のチェックサムを求める
	#   戻値: [string]0～65535 ※0埋めなし
	#   ※ 参考 UbuntuのsumコマンドのBSDアルゴリズムに限り0埋めありの結果を返してくる
	#   ※ 注意 チェックサムファイル内の数値は0埋めされている場合あり 比較時は注意
	static [string] CalcSumBsd([System.IO.Stream]$stream){
		return cs-CalcSumBsd $stream;
	}


	##################################################################
	# sumコマンド(System Vアルゴリズム)準拠のチェックサムを求める
	#   戻値: [string]0～65535 ※0埋めなし
	#   ※ 注意 チェックサムファイル内の数値は0埋めされている可能性を考える 比較時は注意
	static [string] CalcSumSysv([System.IO.Stream]$stream){
		return cs-CalcSumSysv $stream;
	}


	##################################################################
	# ストリームのハッシュ値を得る
	#   引数1: stream
	#   引数2: [string]ハッシュアルゴリズム
	#   戻値: @{hash:[string]ハッシュ値; unixname:[string]表示用アルゴリズム名; size:[int]データサイズ;}
	#   ※ sizeはアルゴリズムに適した数値。サイズ計算が不要なアルゴリズムの場合は-1が入る。
	static [Hashtable] GetHtHash([System.IO.Stream]$stream, $algo){
		$strHash = "";
		[int64]$intDataSize = -1;

		# Get-FileHashで処理できるもの
		if($Script:ALGO_NAMES[$algo]["msparam"] -ne ""){
			$strHash = (Get-FileHash -InputStream $stream -Algorithm $Script:ALGO_NAMES[$algo]["msparam"]).Hash;

		# 自力算出のアルゴリズム
		}else{
			if($algo -eq "cksum"){
				$strHash = [LoSum]::CalcCkSum($stream);
				$intDataSize = $stream.Length;
			}elseif($algo -eq "sumbsd"){
				$strHash = [LoSum]::CalcSumBsd($stream);
				$intDataSize = [int64] [Math]::Ceiling( ($stream.Length)/1024 ); # 1KiB単位・小数切り上げ
			}elseif($algo -eq "sumsysv"){
				$strHash = [LoSum]::CalcSumSysv($stream);
				$intDataSize = [int64] [Math]::Ceiling( ($stream.Length)/512 ); # 512B単位・小数切り上げ
			}
		}

		return @{
			hash     = $strHash.ToLower($Script:NONCUL);
			unixname = $Script:ALGO_NAMES[$algo]["unixname"];
			size     = $intDataSize;
		};
	}

	##################################################################
	# ファイルのハッシュ値を得る
	#   引数1: File
	#   引数2: [string]ハッシュアルゴリズム
	#   戻値: @{hash:[string]ハッシュ値; unixname:[string]表示用アルゴリズム名; size:[int]データサイズ;}
	#   ※ sizeはアルゴリズムに適した数値。サイズ計算が不要なアルゴリズムの場合は-1が入る。
	static [Hashtable] GetHtHash([System.IO.FileInfo]$file, $algo){
		$fs = [System.IO.FileStream]::new($file, [System.IO.FileMode]::Open, [System.IO.FileAccess]::Read, [System.IO.FileShare]::ReadWrite + [System.IO.FileShare]::Delete);

		$ht = @{};
		try{
			$ht = [LoSum]::GetHtHash($fs, $algo);
		}finally{
			$fs.Close();
		}
		return $ht;
	}


	##################################################################
	# ファイル名指定用文字列を与えると、FileSystemInfoと表示用ファイルパスのハッシュテーブル配列を得る
	#   引数1: ユーザ入力のファイル名指定用文字列 "./*.*" "ファイル名*" "ファイル名.txt" "c:\foo*" など
	#   引数2: 再帰検索するか（再帰検索時はディレクトリを含む文字列を指定できない。外部であらかじめチェックせよ）
	#   戻値: @{fsinfo:[FileSystemInfo]; viewname:[string]表示用ファイル名;} の配列
	#   ※ 該当ファイル0個の場合、長さ0の配列を返す
	#   ※ viewnameは入力形式に応じて絶対パス・相対パス・冒頭に.\がつくかどうか などが変化する。
	#   ※ パス区切り文字は "/"
	static [Object] SearchFile($strSentence, $isRecurse=$false){
		$strSentence = $strSentence -replace "/", "\"; # 全ての処理は"\"で行い最後に"/"に変換 (コマンドレットは常に\を吐くため)

		$isDotStarted   = $false; # 入力文字列の冒頭に ".\" or "..\" があるか
		$isAbsolutePath = $false; # 入力文字列は絶対パスか

		if(($strSentence -match "^[/\\]") -or ($strSentence -match "^[a-z]:")){
			$isAbsolutePath = $true;
		}elseif($strSentence -match "^\.{1,2}\\"){
			$isDotStarted = $true;
		}

		$arrFsinfoTargets = @();
		$arrFsinfoTargets += Get-ChildItem $strSentence -File -Recurse:$isRecurse;
		if($null -eq $arrFsinfoTargets[0]){ # 該当ファイルなし
			return @();
		}

		$glFileData = New-Object List[Hashtable];
		foreach($fsinfo in $arrFsinfoTargets){
			$strViewName = "";

			if($isAbsolutePath){
				$strViewName = $fsinfo.Fullname;
			}else{
				$strViewName = Resolve-Path -LiteralPath $fsinfo.Fullname -Relative; # LiteralPath必須
				if(!$isDotStarted){
					$strViewName = $strViewName -replace "^\.\\", "";
				}
			}

			$glFileData.add( @{
				fsinfo   = $fsinfo;
				viewname = ($strViewName -replace "\\", "/");
			});
		}

		return [array]$glFileData;
	}


}


# --------------------------------------------------------------------
# ヘルプ表示モード
if($help){
	Write-Output ([LoSum]::SelectIntlMsg(@{
		default = "en";
		"en"    = $HELP_EN;
		"ja-JP" = $HELP_JA_JP;
	}));

	exit;
}


# バージョン表示モード
if($isVersionMode){
	Write-Output "${COM_NAME} (${COM_AUTHER}) ${COM_VER1} ${COM_VER2}";
	exit;
}


# パイプラインが空 かつ ファイル名無指定 かつ カレント再帰検索モードではない かつ ベリファイモードではない
if(!$isRecurse -and ($strSentence -eq "") -and ($null -eq $stdin) -and !$isVerifyMode){
	Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
		default = "en";
		"en"    = "${COM_NAME}: Input FILE(s) or option(s)";
		"ja-JP" = "${COM_NAME}: ファイル名またはオプションを指定して下さい";
	}));
	exit;
}


# ファイル出力モードの場合、使えない文字が含まれていないかチェック ＆ 出力ファイルフルパスを作成・チェック
$strOutfilePath = "";
if($isFileOutput){
	if( [bool]($INVALID_CHARS | ?{$strOutFile.Contains($_)}) ){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: Invalid filename";
			"ja-JP" = "${COM_NAME}: ファイル名に使えない文字が含まれています";
		}));
		exit;
	}

	if($strOutFile -match "[/\\]$"){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: Invalid filename";
			"ja-JP" = "${COM_NAME}: ファイル名が不正です";
		}));
		exit;
	}

	$strOutfilePath = [LoSum]::GetFullpath($strOutFile);
	if($strOutfilePath -eq ""){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: ${strSumFile}: No such file or directory";
			"ja-JP" = "${COM_NAME}: ${strSumFile}: そのようなファイルやディレクトリはありません";
		}));
		exit;
	}

}


# 照合モード（既存のCHECKSUMファイルを利用して照合する）
if($isVerifyMode){
	if( !(Test-Path $strSumFiles -ErrorAction Continue) ){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: ${strSumFile}: No such file or directory";
			"ja-JP" = "${COM_NAME}: ${strSumFile}: そのようなファイルやディレクトリはありません";
		}));
		exit;
	}

	$sbChecksumText = New-Object System.Text.StringBuilder;
	foreach($file in Get-Item $strSumFiles){
		$sr = New-Object IO.StreamReader((Convert-Path -LiteralPath $file), [Text.Encoding]::UTF8); # フルパス必須
		[void]$sbChecksumText.Append($sr.ReadToEnd());
		$sr.close();
	}

	$strChecksumText = $sbChecksumText.ToString();

	$intCountNotFound   = 0; # ファイル不存在
	$intCountNotMatch   = 0; # ハッシュ値不一致
	$intCountErrorLine  = 0; # ハッシュファイル内のフォーマットエラー行

	foreach($strLine in $strChecksumText -csplit "`r`n|`n|`r"){
		if($strLine -eq ""){
			continue;
		}

		$arrAlgos  = @();
		$strWrittenHash = "";
		$strFilename   = "";
		[Int64]$intWrittenDatasize = -1;

		$str1st = $strLine -replace "\s.*$"; # BSD形式なら"MD5"など、gnu形式ならハッシュ値

		# BSDスタイル: "MD5 (ファイル名) = ハッシュ値"
		if($str1st -match "^(MD5|SHA1|SHA256|SHA384|SHA512)$"){
			foreach($key in $ALGO_NAMES.Keys){
				if($ALGO_NAMES[$key]["unixname"] -eq $str1st){
					$arrAlgos += $key;
				}
			}

			$strWrittenHash = ($strLine -replace "^.*\s").ToLower($NONCUL);
			$strFilename = $strLine -replace "^[shamd1234568]{3,6}\s+\(" -replace "\)\s+=\s+[0-9a-f]{32,128}$";

		# gnuスタイル: "ハッシュ値 *ファイル名" または "ハッシュ値  ファイル名"
		}elseif($str1st -match "^([0-9a-f]{32}|[0-9a-f]{40}|[0-9a-f]{64}|[0-9a-f]{96}|[0-9a-f]{128})$"){
			# ハッシュ値の文字列長で振り分ける
			foreach($key in $ALGO_NAMES.Keys){
				if($ALGO_NAMES[$key]["length"] -eq $str1st.Length){
					$arrAlgos += $key;
				}
			}

			$strWrittenHash = $str1st.ToLower($NONCUL);
			$strFilename = $strLine -replace "^[0-9a-f]{32,128}\s+\*?";

		# sum, cksumスタイル: "10進ハッシュ値 ファイルサイズ値 ファイル名"
		}elseif($strLine -match "^[0-9]{1,10}\s+[0-9]+\s+" -and [Int64]$str1st -le 4294967295){
			$intWrittenHash = [Int64]$str1st;

			$strWrittenHash = [Convert]::ToString($intWrittenHash, 10); # 左0埋めを削除
			$intWrittenDatasize = [Int64]($strLine -replace "^[0-9]+\s+" -replace "\s+.*$");
			$strFilename = $strLine -replace "^[0-9]+\s+[0-9]+\s+";

			if(65535 -lt $intWrittenHash){
				$arrAlgos += "cksum";
			}else{
				$arrAlgos += ("sumbsd", "sumsysv", "cksum");
			}
		}else{
			$intCountErrorLine++;
			continue;
		}

		$strWrittenHash = $strWrittenHash.ToLower($NONCUL);

		# 読み込んだハッシュファイルの行に何らかの異常がある場合
		if(($arrAlgos.Length -eq 0) -or ($strFilename -eq "") -or ($strWrittenHash -eq "")){
			$intCountErrorLine++;
			continue;
		}

		# 対象ファイルが存在しない場合
		if( !(Test-Path -LiteralPath $strFilename -ErrorAction Continue) ){
			$intCountNotFound++;
			Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
				default = "en";
				"en"    = "${COM_NAME}: ${strFilename}: No such file or directory";
				"ja-JP" = "${COM_NAME}: ${strFilename}: そのようなファイルやディレクトリはありません";
			}));
			# Write-Host -fore red "${strFilename}: FAILED open or read"; # ●BSDのmd5sumにはない行
			continue;
		}

		$fsiTarget = Get-Childitem -LiteralPath $strFilename # 対象ファイルのFileSystemInfo
		# ハッシュ値計算・照合
		$isHashMatched = $false; # ハッシュ値照合とファイルサイズ照合に成功
		foreach($algo in $arrAlgos){
			$htCalcedHashdata = [LoSum]::GetHtHash($fsiTarget, $algo);

			$strCalcedHash     = $htCalcedHashdata["hash"];
			$intCalcedDatasize = $htCalcedHashdata["size"];

			if(($strCalcedHash -eq $strWrittenHash) -and ($intCalcedDatasize -eq $intWrittenDatasize)){
				$isHashMatched = [bool]($isHashMatched -bor $true)
				break;
			}else{
				$isHashMatched = [bool]($isHashMatched -bor $false)

			}
		}

		if($isHashMatched){
			Write-Output "${strFilename}: OK";
		}else{
			$intCountNotMatch++;
			Write-Output ([LoSum]::SelectIntlMsg(@{
				default = "en";
				"en"    = "${strFilename}: FAILED";
				"ja-JP" = "${strFilename}: 失敗";
			}));
		}
	}

	if($intCountErrorLine -ne 0){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: WARNING: ${intCountErrorLine} line is improperly formatted";
			"ja-JP" = "${COM_NAME}: 警告: 書式が不適切な行が ${intCountErrorLine} 行あります";
		}));
	}

	if($intCountNotFound -ne 0){ # BSDのmd5sumにはない表示
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: WARNING: ${intCountNotFound} listed file could not be read";
			"ja-JP" = "${COM_NAME}: 警告: 一覧にある ${intCountNotFound} 個のファイルが読み込めませんでした";
		}));
	}

	if($intCountNotMatch -ne 0){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: WARNING: ${intCountNotMatch} computed checksum did NOT match";
			"ja-JP" = "${COM_NAME}: 警告: ${intCountNotMatch} 個の計算したチェックサムが一致しませんでした";
		}));
	}


# ハッシュ作成モード（標準入力のハッシュ）
}elseif($isPipelineMode){
	# （ファイル出力モードの場合）出力ファイルの存在確認
	if( $isFileOutput -and (Test-Path -LiteralPath $strOutfilePath) ){
		if($isWriteForce){
			# 最初にファイルの中身を空にする
			[LoSum]::WriteTextFile("", $strOutfilePath, $true);
		}elseif($isWriteAdd){

		}else{
			Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
				default = "en";
				"en"    = "${COM_NAME}: ${strOutfilePath}: The file already exists";
				"ja-JP" = "${COM_NAME}: ${strOutfilePath}: ファイルは既に存在します";
			}));
			exit;
		}
	}

	# パイプラインからの入力は、UTF8文字列の場合とbyte配列の場合がある。生バイナリはない(PowerShellの仕様により壊される)
	if($stdin -is [byte[]]){
		$ms = [System.IO.MemoryStream]::new($stdin);
	}else{
		$ms = [System.IO.MemoryStream]::new( [Text.Encoding]::UTF8.GetBytes($stdin) );
	}

	$htHashData = [LoSum]::GetHtHash($ms, $strAlgo);
	$ms.Close();

	$strHash         = $htHashData["hash"];
	$strUnixAlgoName = $htHashData["unixname"];
	$intDataSize     = $htHashData["size"];

	$strLine = "";

	if($intDataSize -eq -1){ # データサイズなし(-1)のアルゴリズムの場合、MD5以降用のモダンな書式で出力する
		if($isBsdStyleOutput){
			$strLine = "${strUnixAlgoName} (-) = ${strHash}"
		}else{
			$strLine = "${strHash} ${strBinFlag}-";
		}
	}else{
		$strLine = "${strHash} ${intDataSize} ${strViewPath}";
	}

	Write-Output $strLine;
	if($isFileOutput){
		[LoSum]::WriteTextFile(($strLine + $strSeparater), $strOutfilePath, $false);
	}


# ハッシュ作成モード（ファイルのハッシュ値）
}else{
	# 再帰モードとディレクトリ名指定は併存できない
	if( $isRecurse -and ($strSentence.Contains('\') -or $strSentence.Contains('/') -or ($strSentence -eq ".") -or ($strSentence -eq "..")) ){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: Cannot set directory path in Recurse mode";
			"ja-JP" = "${COM_NAME}: 再帰検索モードではディレクトリ名を含むパスは指定できません";
		}));

		exit;
	}

	# 存在確認 (LiteralPathはつけない)
	if( !(Test-Path $strSentence -ErrorAction Continue) ){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: ${target}: No such file or directory";
			"ja-JP" = "${COM_NAME}: ${target}: そのようなファイルやディレクトリはありません";
		}));
		exit;
	}

	# ディレクトリ単指定は不可
	if( (Test-Path -LiteralPath $strSentence -ErrorAction Continue) -and (Get-Item -LiteralPath $strSentence).PSIsContainer ){
		Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
			default = "en";
			"en"    = "${COM_NAME}: ${target}: Is a directory";
			"ja-JP" = "${COM_NAME}: ${target}: ディレクトリです";
		}));
		exit;
	}

	# （ファイル出力モードの場合）出力ファイルの存在確認
	if( $isFileOutput -and (Test-Path -LiteralPath $strOutfilePath -ErrorAction Continue) ){
		if($isWriteForce){
			# 最初にファイルの中身を空にする
			[LoSum]::WriteTextFile("", $strOutfilePath, $true);
		}elseif($isWriteAdd){

		}else{
			Write-Host -fore red ([LoSum]::SelectIntlMsg(@{
				default = "en";
				"en"    = "${COM_NAME}: ${strOutfilePath}: The file already exists";
				"ja-JP" = "${COM_NAME}: ${strOutfilePath}: ファイルは既に存在します";
			}));
			exit;
		}
	}

	# -zero($isNullCharSep:true)の際、最後に一気に表示するために使う
	$glStrLines = New-Object List[string]; 

	foreach($htFileData in [LoSum]::SearchFile($strSentence, $isRecurse)){
		# ファイル出力モードの場合、出力ファイル自身は除外
		if($strOutfilePath -eq $htFileData["fsinfo"].FullName){
			continue;
		}

		# 表示用のファイルパス
		$strViewPath = $htFileData["viewname"];

		# ハッシュ値等取得
		$htHashData = [LoSum]::GetHtHash($htFileData["fsinfo"], $strAlgo);

		$strHash         = $htHashData["hash"];
		$strUnixAlgoName = $htHashData["unixname"];
		$intDataSize     = $htHashData["size"];

		$strLine = "";
		if($intDataSize -eq -1){ # データサイズなし(-1)のアルゴリズムならMD5以降用のモダンな形式で出力する
			if($isBsdStyleOutput){
				$strLine = "${strUnixAlgoName} (${strViewPath}) = ${strHash}"
			}else{
				$strLine = "${strHash} ${strBinFlag}${strViewPath}";
			}
		}else{
			$strLine = "${strHash} ${intDataSize} ${strViewPath}";
		}

		$glStrLines.add($strLine);

		if(!$isNullCharSep){
			Write-Output $strLine;
		}

		if($isFileOutput){
			[LoSum]::WriteTextFile(($strLine + $strSeparater), $strOutfilePath, $false);
		}
	}

	if($isNullCharSep){
		Write-Output ($glStrLines -join [char]0x00);
	}
}

